/*
  sums() reponds to the "sums" command. It computes the checksum and
  digest for a file and reponds either with

    SUM DIGEST

  if it could compute the sum and the digest, or

    ? CODE MESSAGE

  if an error occurred.
*/

#include "stdincls.h"
#include "types.e"
#include "s-sumfile.e"
#include "print.e"
#include "errcodes.e"
#include "s-store.e"


/* sums -- send the checksum and digest of a file to the client */
EXPORT void sums(FILE *out, const char *root, const char *arg, const DB store)
{
  unsigned char digest[DIGEST_LEN];
  char path[FILENAME_MAX+1];
  uint32_t checksum;
  fileinfo info;
  bool stored;

  /* arg points to the argument of the "sums" command. It may be empty. */
  /* If root is a directory, it ends in '/' */
  if (!make_absolute(root, arg, path)) {
    print(out, "? %03d Path too long\n", EC_TOOLONG);
    return;
  }

  /* See if the store has an entry for this file (likely). */
  stored = store_get(store, arg, &info);
  if (stored) debug("%s found in store\n", path);
  else debug("%s not found in store\n", path);

  /* If the store had an entry and it contained a checksum and digest
     (unlikely), then use them. Otherwise compute them. */
  if (stored && info.status == '+') { /* info contains checksum & digest */
    checksum = info.sums.sum;
    memcpy(digest, info.sums.digest, DIGEST_LEN);
  } else if (!file_checksum_and_digest(path, &checksum, digest)) {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    return;
  }

  /* If the store had an entry without a checksum and digest, add them. */
  if (stored && info.status != '+') {
    info.status = '+';
    info.sums.sum = checksum;
    memcpy(info.sums.digest, digest, DIGEST_LEN);
    (void) store_put(store, info); /* Doesn't matter if it fails */
    free(info.path);		   /* Free no longer needed memory */
    debug("Updated store entry for %s\n", path);
  }

  /* Send the sum and digest to the client. */
  print(out, "%x %s\n", checksum, digest_to_str(digest));
}
