/*
  store_open() is called by serve1() to open the log of the previous
  synchronization between "remote" and "path", or, if there is no log
  yet, to create a new, empty one.

  store_close() is called by serve1() when the log is no longer
  needed, e.g., when it switches to a different remote and path, or is
  about to exit.

  store_put() is called by update() after it updated a file, to store
  the new mode, time and size of the file.

  store_get() is called by change_mode() to get the type of a file and
  by sums() to add a checksum and digest to the entry.

  store_del() is called by update_log() to remove data for a file that
  was deleted.

  store_reset() is called by serve1() after a "reset" command. It
  makes a log empty, i.e., removes all entries.

  cursor_create() and cursor_next() are called by list() to iterate
  over a log and get all information stored in it. Entries added while
  a cursor is in use may or may not be returned by cursor_next().
  Deleting entries may cause the cursor to miss other entries.

  Implementation (log format version 1):

  The log is a text file, which is read into a hash table in memory
  by store_open() and written back by store_close().

  When the log is empty when store_close() is called (e.g., after a
  "reset" command, option "-r"), the log is not written to disk and
  any old log file on disk is deleted. store_close() does not write a
  log with no entries.

  The log is stored in ~/.r2sync/nnnnnn.store, where nnnnnn is a hash
  of the values of "remote" and "path". The hash is a "long int",
  i.e., a 64-bit number on 64-bit machines and a 32-bit number on
  32-bit machines. I.e., log files are not portable.

  The text file starts with the magic string R2SYNC, a space, the
  version of the log format, in this case "1", and a newline. The
  second line is the value of "path" and the third the value of
  "remote".

  Because the number nnnnnn can in rare cases be the same for
  different values of remote and path, store_open() may have to try
  several files, incrementing the number each time, until it finds one
  with the right remote and path, or finds an unused file name.

  The rest of the file conists of lines with a file mode in octal, a
  last modification time in decimal (the number of seconds since the
  epoch, 1 Jan 1970), a file size in bytes, and a path.

  cursor_next() return a pointer to a string stored in the hash
  table. That string should not be modified by the caller.
*/

#include "stdincls.h"
#include "types.e"
#include "getline.e"
#include "print.e"
#include "homedir.e"
#include "s-hashfn.e"
#include "s-hash.e"
#include "s-digest.e"

#define R2SYNCSTORE 1		/* Format of the store files */

struct DB {
  FILE *f;
  char *filename, *root1, *root2;
  hashtable table;		/* path -> fileinfo */
  hashtable sumtable;		/* sum + digest -> path */
};

struct DB_cursor {
  hashtable table;
  hashcursor cursor;
};

EXPORT typedef struct DB *DB;
EXPORT typedef struct DB_cursor *DB_cursor;


/* sum_hash -- compute a hash for a sum + digest */
static unsigned long sum_hash(const void *s, const size_t slen)
{
  return ((suminfo*)s)->sum;	/* Just use the sum, it's good enough */
}


/* sum_equal -- check if two sum+digest keys are the same */
static bool sum_equal(const void *a, const size_t alen,
		      const void *b, const size_t blen)
{
  suminfo *p = (suminfo*)a, *q = (suminfo*)b;

  assert(alen == sizeof(suminfo));
  return p->sum == q->sum && memcmp(p->digest, q->digest, DIGEST_LEN) == 0;
}


/* store_get -- search the log for data about a file */
EXPORT bool store_get(DB db, const char * const path, fileinfo *info)
{
  fileinfo *data;
  size_t datalen;

  if ((data = hash_get(db->table, path, strlen(path)+1, &datalen)))
    *info = *data;
  return data != NULL;
}


/* store_get_by_sum -- search the log for a file with a given sum & digest */
EXPORT bool store_get_by_sum(DB db, const suminfo *sums, fileinfo *info)
{
  size_t len;
  char *p;

  if (!(p = hash_get(db->sumtable, sums, sizeof(*sums), &len))) return false;
  else return store_get(db, p, info);
}


/* store_put -- update a record about a file in the log */
EXPORT bool store_put(DB db, const fileinfo info)
{
  size_t n = strlen(info.path) + 1;
  fileinfo data = info;

  if (!(data.path = strdup(info.path))) return false;
  if (!hash_put(db->table, data.path, n, &data, sizeof(data))) return false;
  if (info.status != '+') return true;
  return hash_put(db->sumtable, &data.sums, sizeof(data.sums), data.path, n);
}


/* store_del -- delete the record for "path" in the log */
EXPORT void store_del(DB db, const char * const path)
{
  size_t plen = strlen(path) + 1, dlen;
  fileinfo *data;

  if ((data = hash_get(db->table, path, plen, &dlen))) {
    free(data->path);
    if (data->status == '+')
      hash_del(db->sumtable, &data->sums, sizeof(data->sums));
    hash_del(db->table, path, plen);
  }
}


/* cursor_create -- create a cursor to iterate over the entries */
EXPORT DB_cursor cursor_create(DB db)
{
  DB_cursor c;

  if (!(c = malloc(sizeof(*c)))) return NULL;
  c->table = db->table;
  if (!(c->cursor = hash_cursor(c->table))) {free(c); return NULL;}
  return c;
}


/* cursor_next -- return entries one by one */
EXPORT bool cursor_next(DB_cursor cursor, fileinfo *info)
{
  fileinfo *data;
  size_t plen, dlen;
  char *p;

  if (!(p = hash_next(cursor->cursor, &plen))) free(cursor); /* The end */
  else if ((data = hash_get(cursor->table, p, plen, &dlen))) *info = *data;
  else assert(!"Cannot happen!");

  return p != NULL;
}


/* store_reset -- remove all entries in a log */
EXPORT void store_reset(DB db)
{
  /* TODO: Free memory allocated for info.path fields */
  /* Delete the tables and create new, empty ones. */
  hash_destroy(db->sumtable);
  db->sumtable = hash_create(sum_hash, sum_equal);

  hash_destroy(db->table);
  db->table = hash_create(NULL, NULL);
}


/* store_close -- write the log to disk, free memory */
EXPORT bool store_close(DB db)
{
  size_t plen, dlen;
  hashcursor c;
  fileinfo *d;
  char *path;

  /* Create a cursor and get the first entry. */
  if (!(c = hash_cursor(db->table))) return false;
  path = hash_next(c, &plen);
  errno = 0;

  if (!path) {

    /* The store is empty. Remove the file instead of rewriting it. */
    remove(db->filename);
    (void) fclose(db->f);

  } else {

    char tmpname[FILENAME_MAX];
    FILE *f;

    /* Write the log to a temporary file. */
    if (!(f = create_temp_file(db->filename, tmpname))) {
      warn(_("Failed to write log. (Cannot create temporary file.)"));
    } else {
      (void) print(f, "R2SYNC %d\n", R2SYNCSTORE);
      (void) print(f, "%s\n", db->root1);
      (void) print(f, "%s\n", db->root2);

      while (path) {
	if ((d = hash_get(db->table, path, plen, &dlen)))
	  /* Can only fail because of an interrupt */
	  print(f, "%o %ld %lld %s\n", d->mode, d->time, d->size, path);
	else
	  warn(_("Failed to write log for %s"), path);
	store_del(db, path);	/* Free allocated memory */
	path = hash_next(c, &plen);
      }

      /* Rename the temporary file, replacing the old log. */
      if (fflush(f) == -1 ||
	(unlink(db->filename) == -1 && errno != ENOENT) ||
	rename(tmpname, db->filename) == -1 ||
	fclose(f) == -1)
	warn(_("Failed to write log. (Cannot replace old log.)"));
    }
  }

  hash_destroy(db->sumtable);
  hash_destroy(db->table);
  free(db->filename);
  free(db->root1);
  free(db->root2);
  free(db);

  /* Return whether the file rewriting was successful. */
  return errno == 0;
}


/* store_open -- open or create the log of a sync between remote and path */
EXPORT DB store_open(char *remote, char *path, bool *created)
{
  char *home, *s, *line, c;
  unsigned long h;
  fileinfo data;
  size_t n;
  DB db;
  int i;

  /* Get the user's home directory */
  if (!(home = home_dir())) return NULL;

  /* Create a string to hold the store's path: ~/.r2sync/nnnnnn.store */
  n = strlen(home) + sizeof(R2SYNCDIR) + 28;
  if (!(s = malloc(n))) {free(home); return NULL;}

  /* Create directory ~/.r2sync if it doesn't exist. */
  sprintf(s, "%s/%s/", home, R2SYNCDIR);
  if (!create_directories(s)) {free(home); free(s); return NULL;}

  /* Create a DB structure in memory */
  if (!(db = malloc(sizeof(*db)))) {free(home); free(s); return NULL;}
  db->f = NULL;
  db->filename = s;
  db->root1 = db->root2 = NULL;
  if (!(db->table = hash_create(NULL, NULL))) {
    free(home); free(s); free(db); return NULL;
  } else if (!(db->sumtable = hash_create(sum_hash, sum_equal))) {
    hash_destroy(db->table); free(home); free(s); free(db); return NULL;
  } else if (!(db->root1 = strdup(path)) || !(db->root2 = strdup(remote))) {
    store_close(db); free(home); return NULL;
  }

  /* Repeatedly construct a file name and see if it contains our
     store. The file name is a hash of remote and path. If it is not
     the right log for remote and path, increase the number by the
     HASHSEED and try again. */
  h = hash(hash(hash(HASHSEED, remote), "\t"), path);
  do {
    /* Close file opened in previous loop */
    if (db->f) {(void) fclose(db->f); db->f = NULL;}

    /* Construct the file name ~/.r2sync/nnnnnn.store */
    sprintf(s, "%s/%s/%lu.store", home, R2SYNCDIR, h);
    h += HASHSEED;		/* File name for the next iteration */

    /* Open the file or create it */
    if ((db->f = fopen(s, "rb+"))) *created = false;
    else if ((db->f = fopen(s, "wb+"))) *created = true;
    else {store_close(db); free(home); return NULL;}

    /* If we just created the file, use it. Otherwise, use it if it
       starts with the right version, remote and path. */
  } while (!*created &&
	   (!(line = getline_chomp(db->f)) ||
	    strncmp(line, "R2SYNC ", 7) != 0 ||
	    strtol(line + 7, NULL, 10) != R2SYNCSTORE ||
	    !(line = getline_chomp(db->f)) ||
	    strcmp(line, path) != 0 ||
	    !(line = getline_chomp(db->f)) ||
	    strcmp(line, remote) != 0));

  /* Free no longer needed strings */
  free(home);

  /* Fail if another r2sync is already using the file. */
  if (flock(fileno(db->f), LOCK_EX | LOCK_NB) == -1) {
    store_close(db); return NULL;
  }

  /* Load file into memory, each line is parsed and becomes one entry. */
  data.status = '-';		/* Indicate that data.sums is empty. */
  while ((line = getline_chomp(db->f)))
    if (sscanf(line, "%o %ld %lld%c%n",
	       &data.mode, &data.time, &data.size, &c, &i) != 4 ||
	c != ' ' || (data.path = line + i, !store_put(db, data))) {
      store_close(db);
      return NULL;
    }

  /* Check that we are at eof and not at an error */
  if (ferror(db->f)) {store_close(db); return NULL;}

  /* Return the in-memory store we just created */
  return db;
}
