/*
  init_progress_meter() is called by client() to get a timestamp and
  the total number of files to transfer. This info is passed on to
  synchronize() and delete_files() to show a progress meter.

  progress_meter() is called by synchronize1() before it starts
  synchronizing and by delete_list() before it deletes a file. It
  returns a static string with an estimate of the remaining time. It
  then subtracts one from the remaining files.

  status() is called by synchronize1() and delete_list() to display a
  line, not wider than the terminal, with the current operation, the
  name of the file and a progress meter. The name of the file will be
  displayed with an ellipsis in the middle if it doesn't fit on a
  terminal line.

  Implementation:

  The estimated time is based only on the total size of all files that
  still need to be synchronized and the total size of all files
  synchronized so far. I.e., the function assumes the files yet to
  update have changed in similar ways as the ones already updated. It
  also does not take into account in which direction files are
  updated, e.g., if host1 is slower than host2 or if the network is
  asymmetric.

  status() calls ioctl() with TIOCGWINSZ to get the current window
  size and installs a SIGWINCH signal handler to be informed when the
  window changed and it needs to call ioctl() again.

  The signal handler sets the global variable window_width to 0 to
  indicate that the value has just been invalidated.

  status() displays the 3rd part (the ETA) against the window's right
  edge, the 1st part (the current operation) against the left edge,
  and the 2nd part (the file name) in the middle. If there is not
  enough room, some part in the middle of the file name is omitted and
  replaced by "...".
*/

#include "stdincls.h"
#include "types.e"
#include "print.e"
#include "c-profile.e"
#include "export.h"

#define DELBYTES 100	      /* How many bytes to count for a deleted file */
EXPORTDEF(DELBYTES)

/* -2 = SIGWINCH handler not initialized
   -1 = window size unknown, probably not a TTY
    0 = window size not yet determined
   >0 = current window size
*/
static int window_width = -2;


/* sigwinch_handler -- react to a SIGWINCH signal */
static void sigwinch_handler(int signal)
{
  window_width = 0; /* Flag that window_width needs to be recalculated */
}


/* init_progress_meter -- initialize time stamp and total bytes to sync */
EXPORT bool init_progress_meter(Profile profile,
				const fileinfo send1[], const int nsend1,
				const fileinfo send2[], const int nsend2,
				const int ndelete)
{
  int i;

  /* Compute the total number of bytes to synchronize */
  profile->nbtotal = 0;
  for (i = 0; i < nsend1; i++) profile->nbtotal += send1[i].size;
  for (i = 0; i < nsend2; i++) profile->nbtotal += send2[i].size;
  profile->nbtotal += ndelete * DELBYTES; /* A deleted file counts as DELBYTES */
  profile->nbtodo = profile->nbtotal;
  profile->nftotal = profile->nftodo = nsend1 + nsend2 + ndelete;
  profile->start = time(NULL);

  /* The following is used by print_statistics() to measure speed */
  profile->sent1 = get_bytes_sent(fileno(profile->to1));
  profile->sent2 = get_bytes_sent(fileno(profile->to2));

  return true;
}


/* progress_meter -- return progress display, then subtract n from bytes to sync */
EXPORT char *progress_meter(Profile p, long long n)
{
  int r1, r2, r;
  time_t t;
  static char x[40];

  if (p->nbtodo != p->nbtotal) {
    t = time(NULL) - p->start;			   /* Seconds since start */
    r1 = p->nbtodo * t / (p->nbtotal - p->nbtodo); /* Remaining based on bytes */
    r2 = p->nftodo * t / (p->nftotal - p->nftodo); /* Remaining based on files */
    r = (r1 + r2)/2.0;				   /* Average of the two */
    if (r > 60 * 60 * 2) sprintf(x, _("%d hour ETA"), (r + 1800) / 3600);
    else if (r > 60 * 2) sprintf(x, _("%d min ETA"), (r + 30) / 60);
    else sprintf(x, _("%d s ETA"), r);
  } else {
    sprintf(x, _("--:-- ETA"));
  }
  assert(p->nbtodo >= n);
  p->nbtodo -= n;
  p->nftodo--;
  return x;
}


/* status -- show a 3-part line of info filling one line of the terminal */
EXPORT void status(const char *start, const char *middle, const char *end)
{
  size_t ellipselen, middle2start, endstart, space1len, space2len, i, j, n;
  size_t cols, startlen, middlelen, middle1len, middle2len, endlen;
  struct sigaction sa;
  struct winsize ws;
  const char *p;
  mbstate_t state;

  assert(start && middle && end);

  /* Is the signal handler installed? */
  if (window_width == -2) {
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    sa.sa_handler = sigwinch_handler;
    (void) sigaction(SIGWINCH, &sa, NULL);
    window_width = 0;	   /* Indicate that we need to call ioctl() */
  }

  /* Do we need to get the window size? */
  if (window_width == 0) {
    if (ioctl(1, TIOCGWINSZ, &ws) == -1) window_width = -1; /* Not a TTY */
    else window_width = ws.ws_col;
  }

  /* Use window_width if known, otherwise just print the info as-is */
  if (window_width > 0) {

    cols = window_width;

    /* Find the length of each string in characters (not bytes) */
    memset(&state, 0, sizeof(state));
    startlen = mbsrtowcs(NULL, &start, 0, &state);
    middlelen = mbsrtowcs(NULL, &middle, 0, &state);
    endlen = mbsrtowcs(NULL, &end, 0, &state);

    /* Determine which part of each string can fit */
    if (endlen <= cols) endstart = 0;
    else {endstart = endlen - cols; endlen = cols;}
    cols -= endlen;
    if (cols > 0 && endlen > 0) {space2len = 1; cols--;} else space2len = 0;

    if (startlen > 0 && cols > 0) {space1len = 1; cols--;} else space1len = 0;
    if (startlen > cols) startlen = cols;
    cols -= startlen;

    if (middlelen <= cols) {	/* middle fits completely */
      middle1len = middlelen;
      middle2len = 0;
      middle2start = middlelen;
      ellipselen = 0;
      space2len += cols - middlelen;
    } else {			/* Split middle and insert an ellipsis */
      ellipselen = 3;
      if (ellipselen > cols) ellipselen = cols;
      cols -= ellipselen;
      middle2len = cols / 2;
      middle2start = middlelen - middle2len;
      middle1len = (cols + 1) / 2;
    }

    /* Write each part to stdout */
    for (i = 0, p = start; i < startlen; i++, p += n) {
      n = mbrtowc(NULL, p, 100, &state);
      for (j = 0; j < n; j++) putchar(p[j]);
    }
    if (space1len) putchar(' ');

    memset(&state, 0, sizeof(state));
    for (i = 0, p = middle; i < middle1len; i++, p += n) {
      n = mbrtowc(NULL, p, 100, &state);
      for (j = 0; j < n; j++) putchar(p[j]);
    }
    for (; i < middle2start; i++, p += n) /* Skip central part of middle */
      n = mbrtowc(NULL, p, 100, &state);
    for (i = 0; i < ellipselen; i++) putchar('.');
    for (i = 0; i < middle2len; i++, p += n) {
      n = mbrtowc(NULL, p, 100, &state);
      for (j = 0; j < n; j++) putchar(p[j]);
    }
    for (i = 0; i < space2len; i++) putchar(' ');

    memset(&state, 0, sizeof(state));
    for (i = 0, p = end + endstart; i < endlen; i++, p += n) {
      n = mbrtowc(NULL, p, 100, &state);
      for (j = 0; j < n; j++) putchar(p[j]);
    }

  } else
    printf("%s %s %s\r", start, middle, end);
}
