/* flactags -- print comments (Vorbis tags) according to a format string
 *
 * Inspired by mp3info
 *
 * To do:
 *
 * - Handle embedded \0 characters, rather than end the string there.
 *
 * - Free memory, such as that allocated by encode()
 *
 * - Use more portability things from config.h
 *
 * Author: Bert Bos <bert@w3.org>
 * Created: 22 May 2016
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>
#include <sysexits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <ctype.h>
#include <math.h>
#include <locale.h>
#include <iconv.h>
#include <getopt.h>
#include <assert.h>
#include <FLAC/format.h>
#include <FLAC/metadata.h>

#if defined HAVE_ICONV || defined HAVE_LIBICONV
# define USAGE "[-e|-e1|-e2|-e3] [-R] [-f FORMAT] [-h] [-V] file [file...]"
#else
# define USAGE "[-e|-e1|-e2|-e3] [-f FORMAT] [-h] [-V] file [file...]"
#endif

static const struct option longopts[] = {
  {"escapes", optional_argument, NULL, 'e'},
#if defined HAVE_ICONV || defined HAVE_LIBICONV
  {"raw", no_argument, NULL, 'R'},
#endif
  {"format", required_argument, NULL, 'f'},
  {"help", no_argument, NULL, 'h'},
  {"version", no_argument, NULL, 'V'},
};

static int use_escapes = 0;	/* Whether option -e was given */
static char *format = NULL;	/* Option -f */
#if defined HAVE_ICONV || defined HAVE_LIBICONV
static int use_raw = 0;		/* Whether option -R was given */
static iconv_t iconv_to_utf_8;
static iconv_t iconv_to_local;
#endif

typedef struct _stringbuf {
  char *s;
  size_t len, buflen;
} stringbuf;


/* xval -- convert a digit to its value as an int */
#define xval(c) ('0' <= (c) && (c) <= '9' ? (c) - '0' : tolower(c) - 'a' + 10)

/* htoi -- convert a 2-digit hexadecimal number to int */
#define htoi(s) (16 * xval(*(s)) + xval(*((s)+1)))

/* isoct -- true if the character is in '0'..'7' */
#define isoct(c) ('0' <= (c) && (c) <= '7')

/* otoi -- convert 3-digit octal number to int */
#define otoi(s) (64 * xval(*(s)) + 8 * xval(*(s+1)) + xval(*(s+2)))


/* stringbuf_init -- initialize a stringbuf to the empty string */
static void stringbuf_init(stringbuf *b)
{
  b->s = NULL;
  b->len = b->buflen = 0;
}


/* stringbuf_as_string -- return the string (char *) of a stringbuf */
static char *stringbuf_as_string(stringbuf *b)
{
  if (b->len == b->buflen) {
    if (!(b->s = realloc(b->s, b->buflen + 1))) err(EX_OSERR, NULL);
    b->buflen += 1;
  }
  b->s[b->len] = '\0';
  return b->s;
}


#if defined HAVE_ICONV || defined HAVE_LIBICONV
/* encode -- return a copy of a string, with character encoding changed */
static char *encode(iconv_t converter, char *s)
{
  char *t, *h;
  size_t i, j, n;

  assert(s);

  if (converter == (iconv_t)(-1)) return s; /* Encoding unavailable */

  i = strlen(s);
  j = 5 * i + 2;

  h = t = malloc(j);
  if (!t) err(EX_OSERR, NULL);

  n = iconv(converter, &s, &i, &h, &j);
  if (n == (size_t)(-1)) err(EX_DATAERR, NULL);
  /* To do: check for errno == E2BIG instead of allocating so much */

  if (j > 0) *h = '\0';
  if (j > 1) *(h+1) = '\0';
  return t;
}

/* to_utf_8 -- return a copy of a local string re-encoded as UTF-8 */
static char *to_utf_8(char *s) {return encode(iconv_to_utf_8, s);}

/* to_local -- return a copy of a UTF-8 string re-encoded in local encoding */
static char *to_local(char *s) {return encode(iconv_to_local, s);}

#else /* !HAVE_LIBICONV */

/* to_utf_8 -- return a copy of a string */
static char *to_utf_8(char *s) {return strdup(s);}

/* to_local -- return a copy of a string */
static char *to_local(char *s) {return strdup(s);}

#endif /* HAVE_LIBICONV */


/* addc -- add a character to a stringbuf */
static void addc(char c, stringbuf *b)
{
  if (b->len == b->buflen) {
    if (!(b->s = realloc(b->s, b->buflen + 200))) err(EX_OSERR, NULL);
    b->buflen += 200;
  }
  b->s[b->len] = c;
  b->len++;
}


/* adds -- add a string to a stringbuf */
static void adds(const char *s, stringbuf *b)
{
  assert(s);
  for (; *s; s++) addc(*s, b);
}


/* add_escaped -- append a string with certain characters escaped */
static void add_escaped(const char *s, stringbuf *b)
{
  if (s)
    for (; *s; s++)
      switch (*s) {
      case '\n': adds("\\n", b); break;
      case '\r': adds("\\r", b); break;
      case '\\': adds("\\\\", b); break;
      case '\0': adds("\\0", b); break;
      case '\t': adds(use_escapes & 1 ? "\\t" : "\t", b); break;
      case '"': adds(use_escapes & 2 ? "\"\"" : "\"", b); break;
      default: addc(*s, b);
      }
}


/* addtag -- append the value(s) for a given tag */
static void addtag(const FLAC__StreamMetadata_VorbisComment *vc,
		   const char *tag, stringbuf *b)
{
  int n = vc->num_comments, i, j;
  size_t taglen = strlen(tag);
  FLAC__StreamMetadata_VorbisComment_Entry *comments = vc->comments;

  for (j = 0, i = 0; i < n; i++) {
    if (strncasecmp((char*)comments[i].entry, tag, taglen) == 0 &&
	comments[i].entry[taglen] == '=') {
      if (j != 0) adds("; ", b);
      j++;
      if (use_escapes) add_escaped((char*)comments[i].entry + taglen + 1, b);
      else adds((char*)comments[i].entry + taglen + 1, b);
    }
  }
}


/* print_formatted -- print according to the given format string on stdout */
static void print_formatted(char *fname,
			    const FLAC__StreamMetadata *tags,
			    const FLAC__StreamMetadata *info)
{
  const char *basename;
  char h[25], *p, *q;
  const char *s;
  struct stat b;
  stringbuf r;
  const FLAC__StreamMetadata_VorbisComment *vc = &tags->data.vorbis_comment;
  const FLAC__StreamMetadata_StreamInfo *vi = &info->data.stream_info;
    
  /*
   * To do: approximate %G (genre number) with genres from MP3:
   * http://www.id3.org/id3v2.3.0#head-129376727ebe5309c1de1888987d070288d7c7e7
   */

  stringbuf_init(&r);
  fname = to_utf_8(fname);
  basename = strrchr(fname, '/');
  if (!basename) basename = fname; else basename++;

  for (s = format; *s; s++)
    if (*s == '\\')
      switch (*(++s)) {
      case 'n': addc('\n', &r); break;
      case 't': addc('\t', &r); break;
      case 'v': addc('\v', &r); break;
      case 'b': addc('\b', &r); break;
      case 'r': addc('\r', &r); break;
      case 'f': addc('\f', &r); break;
      case 'a': addc('\a', &r); break;
      case '\\': addc('\\', &r); break;
      case 'x':
	if (!isxdigit(*(s+1)) || !isxdigit(*(s+2))) {addc('\\', &r); s--;}
	else {addc(htoi(s + 1), &r); s += 2;}
	break;
      case '0': case '1': case '2': case '3':
      case '4': case '5': case '6': case '7':
	if (!isoct(*(s+1)) || !isoct(*(s+2))) {addc('\\', &r); s--;}
	else {addc(otoi(s), &r); s += 2;}
	break;
      case '\0': addc('\\', &r); s--; break;
      default: addc('\\', &r); s--;
      }
    else if (*s == '%')
      switch (*(++s)) {
      case '%': addc('%', &r); break;
      case '{':
	if (!(p = strchr(s + 1, '}'))) errx(EX_USAGE, "'%%{' without '}'");
	else if (!(q = malloc(p - s))) err(EX_OSERR, NULL);
	else {strncpy(q, s+1, p-s-1); q[p-s-1] = '\0'; addtag(vc, q, &r); s= p;}
	break;
      case 'f': adds(basename, &r); break;
      case 'F': adds(fname, &r); break;
      case 'k':
	(void) stat(fname, &b);	/* To do: errors... */
	/* st_size is long int or long long int on different systems */
	sprintf(h, "%lld", (long long int)b.st_size / 1024);
	adds(h, &r);
	break;
      case 'a': addtag(vc, "ARTIST", &r); break;
      case 'c': addtag(vc, "DESCRIPTION", &r); break;
      case 'g': addtag(vc, "GENRE", &r); break;
      case 'l': addtag(vc, "ALBUM", &r); break;
      case 'n': addtag(vc, "TRACKNUMBER", &r); break;
      case 't': addtag(vc, "TITLE", &r); break;
      case 'y': addtag(vc, "DATE", &r); break;
      case 'C': addtag(vc, "COPYRIGHT", &r); break;
      case 'L': case 'v': adds((char*)vc->vendor_string.entry, &r); break;
      case 'O': addtag(vc, "SOURCEMEDIA", &r); break;
      case 'o': sprintf(h, "%d", vi->channels); adds(h, &r); break;
      case 'Q': sprintf(h, "%u", vi->sample_rate); adds(h, &r); break;
      case 'q': sprintf(h, "%f", 0.001*vi->sample_rate); adds(h, &r); break;
      case 'r':
	sprintf(h, "%f", 0.001*vi->sample_rate*vi->bits_per_sample);
	adds(h, &r);
	break;
      case 'm':
	sprintf(h, "%llu",(long long unsigned)vi->total_samples/vi->sample_rate/60);
	adds(h, &r);
	break;
      case 's':
	sprintf(h, "%f", fmod((float)vi->total_samples/vi->sample_rate, 60));
	adds(h, &r);
	break;
      case 'S':
	sprintf(h, "%f", (float)vi->total_samples/vi->sample_rate);
	adds(h, &r);
	break;
      case '\0': addc('\\', &r); s--; break;
      case 'G': break; /* "musical genre number" */
      case 'e': break; /* "emphasis" */
      case 'E': break; /* "CRC error correction" */
      case 'p': break; /* "padding" */
      case 'u': break; /* "number of good audio frames" */
      case 'b': break; /* "number of corrupt audio frames" */
      default: addc('%', &r); addc(*s, &r);
      }
    else
      addc(*s, &r);

  if (use_raw) fputs(stringbuf_as_string(&r), stdout);
  else fputs(to_local(stringbuf_as_string(&r)), stdout);
}


/* list_tags -- list all the tags from the given file on stdout */
static void list_tags(const FLAC__StreamMetadata *tags)
{
  stringbuf result;
  FLAC__uint32 i;
  const FLAC__StreamMetadata_VorbisComment *comments =
    &tags->data.vorbis_comment;
 
  assert(tags->type == FLAC__METADATA_TYPE_VORBIS_COMMENT);
  stringbuf_init(&result);

  for (i = 0; i < comments->num_comments; i++) {
    if (use_escapes) add_escaped((char*)comments->comments[i].entry, &result);
    else adds((char*)comments->comments[i].entry, &result);
    addc('\n', &result);
  }
  if (use_raw) fputs(stringbuf_as_string(&result), stdout);
  else fputs(to_local(stringbuf_as_string(&result)), stdout);
}


/* process -- handle one file */
static void process(char *filename)
{
  FLAC__StreamMetadata *tags, info;

  if (!FLAC__metadata_get_tags(filename, &tags))
    warnx("%s: Cannot read FLAC file", filename);
  else if (!FLAC__metadata_get_streaminfo(filename, &info))
    warnx("%s: Cannot read FLAC file", filename);
  else {
    if (format) print_formatted(filename, tags, &info); else list_tags(tags);
    FLAC__metadata_object_delete(tags);
  }
}


static void usage(const char *progname)
{
  fprintf(stderr, "Usage: %s %s\n", progname, USAGE);
  exit(EX_USAGE);
}


/* help -- print help message */
static void help(const char *progname)
{
  printf("\n\
  Usage:\n\
	%s %s\n\
\n\
  -e, --escapes\n\
	Escape newlines in tag values: CR as \\r, LF as \\n.\n"
#if defined HAVE_ICONV || defined HAVE_LIBICONV
"  -R, --raw\n\
	Output in UTF-8, rather than the character encoding\n\
	of the environment.\n"
#endif
"  -f FORMAT, --format FORMAT\n\
	Don't output a list of tags and values, but output exactly\n\
	the FORMAT string, once for every file. The string may\n\
	contain escapes (\\n, \\t, etc.), one-letter variables\n\
	(%%r, %%F, %%m, %%s, etc.) and long variables (%%{TITLE},\n\
	%%{ARTIST}, etc.).\n\
  -h, --help\n\
	Show this help text and exit.\n\
  -V, --version\n\
	Show version number and exit.\n\
\n", progname, USAGE);
  exit(0);
}


int main(int argc, char *argv[])
{
  int c;

#ifdef HAVE_SETLOCALE
  (void) setlocale(LC_ALL, "");
#endif

  while ((c = getopt_long(argc, argv, "e::Rf:hV", longopts, NULL)) != -1)
    switch (c) {
    case 'e': use_escapes = optarg ? atoi(optarg) : 4; break;
#if defined HAVE_ICONV || defined HAVE_LIBICONV
    case 'R': use_raw = 1; break;
#endif
    case 'f': format = optarg; break;
    case 'h': help(argv[0]); break;
    case 'V': printf("%s\n", PACKAGE_STRING); return 0;
    default: usage(argv[0]);
    }

  if (optind == argc || use_escapes < 0 || use_escapes > 4) usage(argv[0]);

#if defined HAVE_ICONV || defined HAVE_LIBICONV
  if ((iconv_to_utf_8 = iconv_open("UTF-8", "")) == (iconv_t)(-1))
    warn(NULL);
  if ((iconv_to_local = iconv_open("//TRANSLIT", "UTF-8")) == (iconv_t)(-1))
    warn(NULL);
  if (format) format = to_utf_8(format);
#endif

  while (optind != argc) process(argv[optind++]);

  return 0;
}
